{========================================================================}
{=                (c) 1995-98 SwiftSoft Ronald Dittrich                 =}
{========================================================================}
{=                          All Rights Reserved                         =}
{========================================================================}
{=  D 01099 Dresden             = Fax.: +49 (0)351-8037944              =}
{=  Loewenstr.7a                = info@swiftsoft.de                     =}
{========================================================================}
{=  Actual versions on http://www.swiftsoft.de/mmtools.html             =}
{========================================================================}
{=  This code is for reference purposes only and may not be copied or   =}
{=  distributed in any format electronic or otherwise except one copy   =}
{=  for backup purposes.                                                =}
{=                                                                      =}
{=  No Delphi Component Kit or Component individually or in a collection=}
{=  subclassed or otherwise from the code in this unit, or associated   =}
{=  .pas, .dfm, .dcu, .asm or .obj files may be sold or distributed     =}
{=  without express permission from SwiftSoft.                          =}
{=                                                                      =}
{=  For more licence informations please refer to the associated        =}
{=  HelpFile.                                                           =}
{========================================================================}
{=  $Date: 08.09.98 - 11:09:20 $                                        =}
{========================================================================}
unit uLaw;

interface

uses MMSystem, MMRegs;

function Linear2uLaw(Sample: Smallint; Zerotrap: Boolean): Byte;
function uLaw2Linear(Sample: Byte): Smallint;

procedure Build_uLawHeader(pwfx: PWaveFormatEx; nChannels: Word; dwSampleRate: Longint);

implementation

(*==============================================================================
   This routine converts from linear to ulaw

   Original by Craig Reese:  IDA/Supercomputing Research Center
              Joe Campbell:  Department of Defense
   29 September 1989
   See http://www.speech.cs.cmu.edu/comp.speech/Section2/Q2.7.html

   References
   1) CCITT Recommenations G.711 (very difficult to follow)
   2) "A New Digital Technique for Implementation of Any
      Continuous PCM Companding Law," Villeret, Michel, et al.
      1973 IEEE Int. Conf. on Communications, Vol. 1, 1973,
      pg. 11.12-11.17
   3) MIL-STD-188-113, "Interoperability and Performance Standards
      for Analog-to-Digital Conversion Techniques,"
      17 February 1987

   Input: Signed 16 bit linear sample
   Output: 8 bit ulaw sample
==============================================================================*)
function Linear2uLaw(Sample: Smallint; Zerotrap: Boolean): Byte;
type
  TExp_LuT = array [0..255] of integer;

const
  BIAS = $84;
  CLIP = 32635;
  Exp_LuT: TExp_LuT = (0,0,1,1,2,2,2,2,3,3,3,3,3,3,3,3,
                       4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,
                       5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
                       5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
                       6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,
                       6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,
                       6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,
                       6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,
                       7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
                       7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
                       7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
                       7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
                       7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
                       7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
                       7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
                       7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7);

var
  Sign, Exponent, Mantissa: integer;

begin
   //  Get the sample into sign-magnitude
   Sign := (Sample shr 8) and $80;  // set aside sign
   if (Sign <> 0) then
       Sample := -Sample;           // get magnitude
   if (Sample > CLIP) then
       Sample := CLIP;              // clip the magnitude

   // Convert from 16 bit linear to ulaw
   Sample   := Sample + BIAS;
   Exponent := Exp_LuT[(Sample shr 7) and $FF];
   Mantissa := (Sample shr (Exponent + 3)) and $0F;
   Result   := not (Sign or (Exponent shl 4) or Mantissa);
   if Zerotrap and (Result = 0) then
      Result := $02;  // optional CCITT trap
end;

(*==============================================================================
   This routine converts from ulaw to 16 bit linear

   Original by Craig Reese:  IDA/Supercomputing Research Center
   29 September 1989
   See http://www.speech.cs.cmu.edu/comp.speech/Section2/Q2.7.html

   References:
   1) CCITT Recommenations G.711 (very difficult to follow)
   2) MIL-STD-188-113, "Interoperability and Performance Standards
      for Analog-to-Digital Conversion Techniques,"
      17 February 1987

   Input: 8 bit ulaw sample
   Output: signed 16 bit linear sample
==============================================================================*)
function uLaw2Linear(Sample: Byte): Smallint;
type
   TExp_LuT = array [0..7] of integer;

const
   Exp_LuT: TExp_LuT = (0, 132, 396, 924, 1980, 4092, 8316, 16764);

var
   Sign, Exponent, Mantissa: integer;

begin
   Sample := not Sample;
   Sign := Sample and $80;
   Exponent := (Sample shr 4) and $07;
   Mantissa := Sample and $0F;
   Result := Exp_LuT[Exponent] + (Mantissa shl (Exponent + 3));
   if (Sign <> 0) then
       Result := -Result;
end;

(*============================================================================*)
procedure Build_uLawHeader(pwfx: PWaveFormatEx; nChannels: Word; dwSampleRate: Longint);
begin
   { fill in the info for our destination format... }
   pwfx^.wFormatTag     := WAVE_FORMAT_MULAW;
   pwfx^.nChannels      := nChannels;
   pwfx^.nSamplesPerSec := dwSampleRate;
   pwfx^.wBitsPerSample := 8;

   { set nAvgBytesPerSec and nBlockAlign }
   pwfx^.nBlockAlign    := nChannels;
   pwfx^.nAvgBytesPerSec:= pwfx^.nBlockAlign * pwfx^.nSamplesPerSec;
   pwfx^.cbSize := 0;
end;

end.
